#!/usr/bin/env bash
set -euo pipefail

# Check CLI availability
if ! command -v ylm2 >/dev/null 2>&1; then
  echo "Error: 'ylm2' not found in PATH. Install Youlean Loudness Meter 2 CLI or add it to PATH."
  read -n 1 -s -p "Press any key to exit..."
  echo
  exit 1
fi

# --- Get folder path safely ---
FOLDER="${1:-}"

if [[ -z "$FOLDER" ]]; then
  read -r -p "Enter analysis folder path: " FOLDER
fi

# Convert any escaped spaces (\ ) into real spaces
FOLDER=$(echo "$FOLDER" | sed 's/\\ / /g')

# Remove surrounding quotes if present
FOLDER="${FOLDER%\"}"
FOLDER="${FOLDER#\"}"

# Debug (optional)
# echo "DEBUG Folder='$FOLDER'"

if [[ ! -d "$FOLDER" ]]; then
  echo "Error: Folder does not exist: $FOLDER"
  read -n 1 -s -p "Press any key to exit..."
  echo
  exit 1
fi

# Build list of files safely into an array
IFS=$'\n' read -r -d '' -a FILES < <(find "$FOLDER" -type f \( -iname "*.mp3" -o -iname "*.flac" -o -iname "*.wav" -o -iname "*.ogg" \) -print0 | xargs -0 -n1 echo && printf '\0')

TOTAL=${#FILES[@]}
echo "Found $TOTAL audio file(s) in: \"$FOLDER\""

if [[ $TOTAL -eq 0 ]]; then
  read -n 1 -s -p "Press any key to exit..."
  echo
  exit 0
fi

processed=0
failed=0

for file in "${FILES[@]}"; do
  echo "Analyzing: \"$file\""
  if ylm2 --input-file-path "$file" --export --export-type "TEXT_SUMMARY"; then
    ((processed++))
  else
    echo "  -> Error analyzing file."
    ((failed++))
  fi
done

echo "Done. Found $TOTAL file(s); analyzed $processed; failed $failed."
read -n 1 -s -p "Press any key to exit..."
echo
